<?php

/* --------------------------------------------------------------
   StyleEditCategorySearchController.inc.php 2019-08-29
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2016 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

/**
 * Class representing a category search controller for StyleEdit
 */
class StyleEditCategorySearchController extends HttpViewController
{
    /**
     * Category reader service
     *
     * @var CategoryReadService
     */
    protected $categoryReadService;
    
    /**
     * Category list item JSON serializer
     *
     * @var CategoryListItemJsonSerializer
     */
    protected $categoryListItemJsonSerializer;
    
    
    /**
     * Initialize
     */
    public function init()
    {
        $this->categoryReadService            = StaticGXCoreLoader::getService('CategoryRead');
        $this->categoryListItemJsonSerializer = MainFactory::create(CategoryListItemJsonSerializer::class);
    }
    
    
    /**
     * Return a category list by searching using the provided search term
     *
     * @return JsonHttpControllerResponse
     */
    public function actionByTerm()
    {
        $result = [];
        
        $term     = $this->_getQueryParameter('term');
        $language = $this->_getQueryParameter('language');
        
        if ($term === null || $language === null) {
            throw new InvalidArgumentException('Invalid or missing parameters. Required parameters: "term" and "language"');
        }
        
        $languageCode = MainFactory::create(LanguageCode::class, new StringType($language));
        
        $searchCondition = CategorySearchCondition::createByArray([
                                                                      'search' => [
                                                                          'should' => [
                                                                              [
                                                                                  'match' => [
                                                                                      'categories.categories_id' => $term
                                                                                  ]
                                                                              ],
                                                                              [
                                                                                  'like' => [
                                                                                      'categories_description.categories_name' => "%{$term}%"
                                                                                  ]
                                                                              ]
                                                                          ]
                                                                      ]
                                                                  ]);
        
        $categories = $this->categoryReadService->searchCategories($languageCode, $searchCondition)->getArray();
        
        foreach ($categories as $category) {
            $data     = json_decode($this->categoryListItemJsonSerializer->serialize($category));
            $result[] = [
                'id'     => $data->id,
                'name'   => $data->name,
                'active' => $data->isActive
            ];
        }
        
        return MainFactory::create(JsonHttpControllerResponse::class, $result);
    }
}